//=============================================================================
//
// LightRun.c:
//
// Example for switching lamps at the outputs O1 to O6 with simple C-Code 
//
//-----------------------------------------------------------------------------
// Disclaimer - Exclusion of Liability
//
// This software is distributed in the hope that it will be useful,but WITHOUT 
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. It can be used an modified by anyone
// free of any license obligations or authoring rights.
//=============================================================================

#include "StdAfx.h"

enum MotorIdx {MOTOR_1 = 0, MOTOR_2, MOTOR_3, MOTOR_4};
enum PwmOut {OUT1 = 0, OUT2, OUT3, OUT4, OUT5, OUT6, OUT7, OUT8};

#define MOTOR_ON    TRUE
#define MOTOR_OFF   FALSE

#define STRMAXLEN   80

HANDLE  fthdl;
DWORD   errCode;
char    ComPortName[STRMAXLEN];
char    LibVersion[STRMAXLEN];


//=============================================================================

/*-----------------------------------------------------------------------------
 *  InitAllOutput  
 *---------------------------------------------------------------------------*/
static void InitAllOutput(int status) {

    //  set all Motor output OFF, PWM output ON or Motor output ON, PWM output OFF
    SetFtMotorConfig(fthdl, TA_LOCAL, MOTOR_1, (status == 1));
    SetFtMotorConfig(fthdl, TA_LOCAL, MOTOR_2, (status == 1));
    SetFtMotorConfig(fthdl, TA_LOCAL, MOTOR_3, (status == 1));
    SetFtMotorConfig(fthdl, TA_LOCAL, MOTOR_4, (status == 1));
}

/*-----------------------------------------------------------------------------
 *  LightRun  
 *---------------------------------------------------------------------------*/
static void LightRun(void) {

    int i, outIdx;
    int duty, wait;

    //  set all Motor output OFF, PWM output ON
    InitAllOutput(MOTOR_OFF);

    duty = 512;
    wait = 200;
    for (i = 0; i < 3; i++, wait /= 2) {

        for (outIdx = 0; outIdx <= OUT6; outIdx++) {
            SetOutPwmValues(fthdl, TA_LOCAL, outIdx, duty);
            Sleep(wait);
            SetOutPwmValues(fthdl, TA_LOCAL, outIdx, 0);
        }

        Sleep(wait);

        for (outIdx = OUT6; outIdx >= OUT1; outIdx--) {
            SetOutPwmValues(fthdl, TA_LOCAL, outIdx, duty);
            Sleep(wait);
            SetOutPwmValues(fthdl, TA_LOCAL, outIdx, 0);
        }

        Sleep(1000);
    }
}

/*-----------------------------------------------------------------------------
 *  CheckParameter  
 *---------------------------------------------------------------------------*/
static int CheckCOMPar(int argc, char *argv[]) {

    char    *pStr;
    int     comNo;

    if (argc >= 2) {
        if (strlen(argv[1]) > 3) {
            if ((pStr=strstr(argv[1],"COM")) != NULL) {
                sscanf(pStr+3, "%d", &comNo);
                if (comNo >= 1 && comNo <= 255)
                    return 0;
                else {
                    printf("LightRun.exe: invalid COM number...\n\n");
                    return 1;
                }
            }
        }
    }

    printf("LightRun.exe: no input given...\n\n");
    return 1;
}

/*-----------------------------------------------------------------------------
 *  main
 *  
 *---------------------------------------------------------------------------*/
int main(int argc, char *argv[]) {

    printf("\nExample LightRun.exe ...\n");

    //  check input paramter
    if (CheckCOMPar(argc,argv)) {
        printf("Usage: LightRun.exe COMxx\t(e.g. COM2 or COM32)\n");
        return 1;
    }

    //  get library version
    ftxGetLibVersionStr(LibVersion, STRMAXLEN);
    printf("\nftMscLib %s\n", LibVersion);

    //  library initialization
    errCode = ftxInitLib();

    strcpy(ComPortName, argv[1]);
    printf("\n\nOpen ComPort '%s' ...\n", ComPortName);

    //  open COM port
    fthdl = ftxOpenComDevice(ComPortName, 38400, &errCode);

    if (errCode == FTLIB_ERR_SUCCESS) {

        printf("Connected to ROBO TX Controller ...\n");

        //  starting Transfer Area
        errCode = ftxStartTransferArea(fthdl);

        if (errCode == FTLIB_ERR_SUCCESS) {

            printf("Transfer Area was started and runs...\n");

            LightRun();

            //  stop Transfer Area
            ftxStopTransferArea(fthdl);
        }

        else {
            //  error case
            printf("Error: Transfer Area was not started !\n");
        }

        //  closing port
        printf("Closing ComPort '%s' ...\n", ComPortName);
        errCode = ftxCloseDevice(fthdl);
    }

    else {
        //  error case
        printf("Error: No interface available (Port '%s')\n", ComPortName);
    }

    //  close library
    ftxCloseLib();

    return 0;
}
